#include "TwitterSony.h"
#include "RankingSony.h"
#include "UserProfileSony.h"
#include "MessagePipe.h"
#include "ErrorCodesSony.h"
#include <tw_dialog.h>

using namespace sce::Toolkit::NP;
using namespace sce::Toolkit::NP::Utilities;

namespace UnityPlugin
{
	Twitter gTwitter;

	PRX_EXPORT bool PrxTwitterIsBusy()
	{
		return gTwitter.IsBusy();
	}

	DO_EXPORT( bool, PrxTwitterGetLastError) (ResultCode* result)
	{
		return gTwitter.GetLastError(result);
	}

	PRX_EXPORT bool PrxTwitterIsDialogOpen()
	{
		return gTwitter.IsDialogOpen();
	}

	PRX_EXPORT ErrorCode PrxTwitterPostMessage(PostTwitter* message)
	{
		return gTwitter.StartDialog(message);
	}

	PRX_EXPORT ErrorCode PrxTwitterCancelDialog()
	{
		return gTwitter.CancelDialog();
	}

	Twitter::Twitter()
		: m_Busy(false)
		, m_DialogOpen(false)
		, m_ActiveDialog(kDlgNone)
		, m_LastResult("Twitter")
	{
	}

	bool Twitter::IsBusy()
	{
		return m_Busy;
	}

	bool Twitter::LoadModules()
	{
		if(sceSysmoduleIsLoaded(SCE_SYSMODULE_NP) != SCE_OK)
		{
			sceSysmoduleLoadModule(SCE_SYSMODULE_NP);
		}

		if(sceSysmoduleIsLoaded(SCE_SYSMODULE_HTTPS) != SCE_OK)
		{
			sceSysmoduleLoadModule(SCE_SYSMODULE_HTTPS);
		}

		if(sceSysmoduleIsLoaded(SCE_SYSMODULE_NET) != SCE_OK)
		{
			sceSysmoduleLoadModule(SCE_SYSMODULE_NET);
		}
		// Also requires that sceSslInit(), sceHttpInit() and sceNpInit() have been called but this should already have happend when npToolkit was initialised.

		// Make sure the photo data directory is mounted (it's ok if it already is).
		int ret = sceAppUtilPhotoMount();
		if(ret != SCE_OK && ret != SCE_APPUTIL_ERROR_BUSY) // Not OK and no already mounted ?
		{
			Messages::LogError("Twitter::%s@L%d - %s", __FUNCTION__, __LINE__, LookupSceErrorCode(ret));
			return false;
		}

		return true;
	}

	ErrorCode Twitter::StartDialog(PostTwitter* message)
	{
		if(!m_DialogOpen && !m_Busy)
		{
			m_LastResult.Reset();

			LoadModules();

			SceInt32 ret = 0;
			SceTwDialogParam twDialogParam;
			SceTwDialogSpecParam twDialogSpecParam;

			// initialize parameter of twitter dialog
			sceTwDialogParamInit(&twDialogParam);
			sceTwDialogSpecParamInit(&twDialogSpecParam);
			twDialogParam.twDialogSpecParam = &twDialogSpecParam;
			twDialogParam.mode = SCE_TW_DIALOG_MODE_DEFAULT;
			if(message->userText && strlen(message->userText) > 0) twDialogParam.twDialogSpecParam->tweetMsgStr = (const SceChar8*)message->userText;
			if(message->imagePath && strlen(message->imagePath) > 0) twDialogParam.twDialogSpecParam->imageFilePath = (const SceChar8*)message->imagePath;
			twDialogParam.twDialogSpecParam->forbidAttachPhoto = message->forbidAttachPhoto;
			twDialogParam.twDialogSpecParam->disableEditTweetMsg = message->disableEditTweetMsg;
			twDialogParam.twDialogSpecParam->forbidOnlyImageTweet = message->forbidOnlyImageTweet;
			twDialogParam.twDialogSpecParam->forbidNoImageTweet = message->forbidNoImageTweet;
			twDialogParam.twDialogSpecParam->disableChangeImage = message->disableChangeImage;
			twDialogParam.twDialogSpecParam->limitToScreenShot = message->limitToScreenShot;

			ret = sceTwDialogInit(&twDialogParam);
			if(ret != SCE_OK)
			{
				return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
			}

			m_ActiveDialog = kDlgTwitter;
			m_DialogOpen = true;
			m_Busy = true;
		}
		else
		{
			Messages::LogWarning("Dialog already open\n");
		}

		return m_LastResult.GetResult();
	}

	ErrorCode Twitter::CancelDialog()
	{
		m_LastResult.Reset();
		int ret = sceTwDialogAbort();
		if(ret != SCE_OK)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}
		return m_LastResult.GetResult();
	}

	void Twitter::UpdateDialog()
	{
		SceCommonDialogStatus		cdStatus;
		SceTwDialogResult			pidResult;
		SceInt32 ret = 0;

		// get twitter dialog status
		cdStatus = sceTwDialogGetStatus();
		switch( cdStatus )
		{
			case SCE_COMMON_DIALOG_STATUS_NONE:
			case SCE_COMMON_DIALOG_STATUS_RUNNING:
				return;
			case SCE_COMMON_DIALOG_STATUS_FINISHED:
				break;
			default:
				Messages::LogWarning("Twitter dialog unknown state %08x", cdStatus);
				return;
		}
		
		// get twitter dialog result
		sceClibMemset( &pidResult, 0x0, sizeof(SceTwDialogResult) );
		ret = sceTwDialogGetResult(&pidResult);
		if(ret != SCE_OK)
		{
			m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
			UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_TwitterDialogCanceled);
			UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_TwitterMessagePostFailed);
			return;
		}

		switch( pidResult.result )
		{
			case SCE_COMMON_DIALOG_RESULT_OK:
				m_Result.result = kDlgOK;
				m_ActiveDialog = kDlgNone;
				m_DialogOpen = false;
				m_Busy = false;
				UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_TwitterDialogFinished);
				UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_TwitterMessagePosted);
				break;

			case SCE_COMMON_DIALOG_RESULT_USER_CANCELED:
				m_Result.result = kDlgCanceled;
				m_ActiveDialog = kDlgNone;
				m_DialogOpen = false;
				m_Busy = false;
				UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_TwitterDialogCanceled);
				break;

			case SCE_COMMON_DIALOG_RESULT_ABORTED:
				m_Result.result = kDlgCanceled;
				m_ActiveDialog = kDlgNone;
				m_DialogOpen = false;
				m_Busy = false;
				UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_TwitterDialogCanceled);
				UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_TwitterMessagePostFailed);
				break;

			case SCE_TW_DIALOG_RESULT_UNAUTHORIZED:
				m_Result.result = kDlgCanceled;
				m_ActiveDialog = kDlgNone;
				m_DialogOpen = false;
				m_Busy = false;
				UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_TwitterDialogCanceled);
				UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_TwitterMessagePostFailed);
				break;

			case SCE_TW_DIALOG_RESULT_FAILED:
				m_Result.result = kDlgCanceled;
				m_ActiveDialog = kDlgNone;
				m_DialogOpen = false;
				m_Busy = false;
				m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
				UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_TwitterDialogCanceled);
				UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_TwitterMessagePostFailed);
				break;


			default:
				m_Result.result = kDlgCanceled;
				m_ActiveDialog = kDlgNone;
				m_DialogOpen = false;
				m_Busy = false;
				UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_TwitterDialogCanceled);
				UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_TwitterMessagePostFailed);
		}

		// terminate twitter dialog
		ret = sceTwDialogTerm();
		if(ret != SCE_OK)
		{
			Messages::LogError("Twitter::%s@L%d - %s", __FUNCTION__, __LINE__, LookupSceErrorCode(ret));
		}
	}

	void Twitter::Update()
	{
		if (m_DialogOpen)
		{
			switch(m_ActiveDialog)
			{
			case kDlgTwitter:
				UpdateDialog();
				break;

			case kDlgNone:
				break;
			}
		}
	}

} // namespace UnityPlugin
